######## training-validation e test con risultati ########

import os
import time
import json
import logging
import pandas as pd
import numpy as np
import matplotlib
import openpyxl
matplotlib.use('Agg')
import matplotlib.pyplot as plt
from sklearn.metrics import accuracy_score, f1_score
from sklearn.metrics import (
    classification_report, confusion_matrix, roc_auc_score,
    roc_curve, precision_recall_curve, balanced_accuracy_score,
    average_precision_score, precision_score, recall_score,
    ConfusionMatrixDisplay
)
import joblib
from sklearn.naive_bayes import GaussianNB
from sklearn.neighbors import KNeighborsClassifier
from sklearn.svm import SVC
from sklearn.tree import DecisionTreeClassifier
from sklearn.ensemble import (
    BaggingClassifier,
    AdaBoostClassifier,
    GradientBoostingClassifier,
    RandomForestClassifier
)
from sklearn.linear_model import SGDClassifier
from sklearn.model_selection import GridSearchCV
from sklearn.model_selection import StratifiedKFold
import seaborn as sns
import shap
from CV_esterna import external_crossval_with_fs

def plot_confusion_matrix(matrix, phase, classifier_name, output_dir):
    disp = ConfusionMatrixDisplay(confusion_matrix=matrix, display_labels=[0, 1])
    disp.plot(cmap=plt.cm.Blues)
    plt.title(f"{classifier_name} - {phase} Confusion Matrix")
    plt.savefig(os.path.join(output_dir, f"{classifier_name}_{phase}_confusion_matrix.png"))
    plt.close()

def plot_roc_curve(y_true, y_prob, phase, classifier_name, output_dir):
    fpr, tpr, _ = roc_curve(y_true, y_prob)
    auc = roc_auc_score(y_true, y_prob)

    plt.figure()
    plt.plot(fpr, tpr, label=f"AUC = {auc:.2f}")
    plt.plot([0, 1], [0, 1], 'k--')
    plt.xlabel('False Positive Rate')
    plt.ylabel('True Positive Rate')
    plt.title(f"{classifier_name} - {phase} ROC Curve")
    plt.legend(loc="lower right")
    plt.savefig(os.path.join(output_dir, f"{classifier_name}_{phase}_roc_curve.png"))
    plt.close()

def plot_precision_recall(y_true, y_prob, phase, classifier_name, output_dir):
    precision, recall, _ = precision_recall_curve(y_true, y_prob)
    ap = average_precision_score(y_true, y_prob)

    plt.figure()
    plt.plot(recall, precision, label=f"AP = {ap:.2f}")
    plt.xlabel("Recall")
    plt.ylabel("Precision")
    plt.title(f"{classifier_name} - {phase} Precision-Recall Curve")
    plt.legend(loc="upper right")
    plt.savefig(os.path.join(output_dir, f"{classifier_name}_{phase}_precision_recall_curve.png"))
    plt.close()

def plot_risultati_test(df_risultati, output_dir):
    # Parametri del grafico
    classificatori = df_risultati['clf_nome']
    x = range(len(classificatori))
    bar_width = 0.25

    # Crea il grafico
    plt.figure(figsize=(12, 6))
    plt.bar([i - bar_width for i in x], df_risultati['F1_score_1'], width=bar_width, label='F1 Score Classe 1', color='skyblue')
    plt.bar(x, df_risultati['roc_auc'], width=bar_width, label='AUC ROC', color='salmon')
    plt.bar([i + bar_width for i in x], df_risultati['bal_accuracy'], width=bar_width, label='Balanced Accuracy', color='lightgreen')

    # Aggiustamenti estetici
    plt.xlabel('Classificatore')
    plt.ylabel('Valore Metrica')
    plt.title('Confronto tra Classificatori')
    plt.xticks(x, classificatori, rotation=45)
    plt.ylim(0, 1.1)
    plt.legend()
    plt.tight_layout()
    plt.grid(axis='y', linestyle='--', alpha=0.6)

    # Salva il grafico
    plt.savefig(output_dir, dpi=300)
    plt.close()
    print(f"Grafico salvato in: {output_dir}")


def plot_and_save_results(results_dict, output_dir, feature_selection_method):
    """
    Crea e salva i grafici per Accuracy e F1-score per ogni combinazione
    di metodo di selezione delle feature e classificatore.

    Parameters:
    - results_dict: dizionario annidato con risultati per classificatori e set (validation/test)
    - output_dir: percorso della cartella dove salvare i grafici
    - feature_selection_method: nome del metodo di selezione delle feature (stringa)
    """

    # Trasforma il dizionario annidato in DataFrame
    rows = []
    for clf_name, clf_results in results_dict.items():
        for set_name in ['train','validation']:
            metrics = clf_results.get(set_name, {})
            row = {
                'Feature Selection': feature_selection_method,
                'Classifier': clf_name,
                'Set': set_name,
                'Accuracy': metrics.get('accuracy', None),
                'ROC AUC': metrics.get('roc_auc', None)
            }
            rows.append(row)

    results_df_flat = pd.DataFrame(rows)

    os.makedirs(output_dir, exist_ok=True)

    for metric_col in ['Accuracy', 'ROC AUC']:
        plt.figure(figsize=(12, 6))
        palette = ["#1f77b4", "#ff7f0e"]
        sns.set_palette(palette)
        sns.barplot(data=results_df_flat, x='Classifier', y=metric_col,
                    hue='Set', hue_order=["train", "validation"])
        plt.title(f'Confronto tra classificatori - {feature_selection_method} ({metric_col})')
        plt.ylabel(metric_col)
        plt.ylim(0, 1)
        plt.xticks(rotation=45)
        plt.legend(title='Set')
        plt.grid(axis='y', linestyle='--', alpha=0.7)
        plt.tight_layout()

        # Salvataggio
        filename = f"{metric_col.replace('-', '').lower()}_comparison.png"
        filepath = os.path.join(output_dir, filename)
        plt.savefig(filepath)
        plt.close()

        print(f"Grafico salvato: {filepath}")

def test_and_SHAP(model, clf_name, feature_selection_name, sel_features, X_test, y_test, output_dir):
    
    print(f"\n[TESTING] Classificatore: {clf_name} | FS: {feature_selection_name}")

    output_dir = os.path.join(output_dir,'risultati_test')
    os.makedirs(output_dir, exist_ok=True)
    
    # Predizione
    y_pred = model.predict(X_test)
    #y_prob = model.predict_proba(X_test)[:, 1]
    test_report = classification_report(y_test, y_pred, output_dict=True)
    matrix = confusion_matrix(y_test, y_pred)
    accuracy = test_report['accuracy']
    f1_score_classe1 = test_report['1.0']['f1-score']
    f1_score_classe0 = test_report['0.0']['f1-score']
    recall_0 = test_report['0.0']['recall']
    recall_1 = test_report['1.0']['recall']
    precision_0 = test_report['0.0']['precision']
    precision_1 = test_report['1.0']['precision']
    bal_accuracy = balanced_accuracy_score(y_test, y_pred)
    
    # Ottieni le probabilità se disponibili
    if hasattr(model, "predict_proba"):
        y_score = model.predict_proba(X_test)[:, 1]
    elif hasattr(model, "decision_function"):
        y_score = model.decision_function(X_test)
    else:
        y_score = None
        
    # Calcolo sicuro dell'AUC
    try:
        auc = roc_auc_score(y_test, y_score) if y_score is not None and len(np.unique(y_test)) > 1 else 'N/A'
    except Exception as e:
        print(f"Errore nel calcolo AUC per {clf_name} in fase TEST: {e}")
        auc = 'N/A'
        
    # Salvataggio report in Excel
    pd.DataFrame(test_report).transpose().to_excel(
        os.path.join(output_dir, f"{clf_name}_TEST_report.xlsx"))
    cm_df = pd.DataFrame(matrix, columns=['Pred 0', 'Pred 1'], index=['True 0', 'True 1'])
    cm_df.transpose().to_excel(os.path.join(output_dir, f"{clf_name}_TEST_confusion_matrix.xlsx"))

    # Salvataggio plot
    plot_confusion_matrix(matrix,'TEST', clf_name, output_dir)
    if y_score is not None and isinstance(auc, float):
        plot_roc_curve(y_test, y_score, 'TEST', clf_name, output_dir)
        plot_precision_recall(y_test, y_score, 'TEST', clf_name, output_dir)
        
    # Calcolo metriche
    risultati_test = {
        'clf_nome': clf_name,
        'test_report': test_report,
        'confusion_matrix': matrix.tolist(),
        'roc_auc': auc,
        'accuracy': accuracy,
        'bal_accuracy': bal_accuracy,
        'F1_score_1': f1_score_classe1,
        'F1_score_0': f1_score_classe0,
        'precision_0': precision_0,
        'precision_1': precision_1,
        'recall_0': recall_0,
        'recall_1': recall_1,
    }
    # ===== SHAP VALUES =====
    try:
        # Usa KernelExplainer se non è un modello tree-based
        if hasattr(model, "predict_proba") and not hasattr(model, "feature_importances_"):
            explainer = shap.Explainer(model.predict_proba, X_test)
        else:
            explainer = shap.Explainer(model, X_test)

        shap_values = explainer(X_test)

        # Imposta i nomi delle colonne se necessario
        if sel_features is not None:
            X_test = X_test.copy()
            X_test.columns = sel_features
        
        try:
            # Se shap_values ha dimensioni (n_samples, n_features, n_classes)
            if shap_values.values.ndim == 3:
                # Estrai solo i valori per la classe 1
                values_classe1 = shap_values.values[:, :, 1]
                base_values_classe1 = shap_values.base_values[:, 1] if shap_values.base_values.ndim == 2 else shap_values.base_values

                shap_values_classe1 = shap.Explanation(
                    values=values_classe1,
                    base_values=base_values_classe1,
                    data=X_test,
                feature_names=X_test.columns
                )
            else:
                # Caso binario: shap_values è già riferito alla classe positiva
                shap_values_classe1 = shap_values
        except Exception as e:
            logging.exception(f"Errore nel processo di training-validation-test del dataset F stats")
            
        # Summary plot SHAP - Classe 1
        shap.summary_plot(shap_values_classe1, X_test, show=False, plot_type="dot")
        shap_path = os.path.join(output_dir, f"{clf_name}_SHAP_summary_classe1.png")
        plt.title(f"SHAP Summary Plot – Classe 1 (positiva)")
        plt.tight_layout()
        plt.savefig(shap_path, dpi=300, bbox_inches='tight')
        plt.close()
        
        # SHAP bar plot
        shap.plots.bar(shap_values_classe1, show=False)
        bar_path = os.path.join(output_dir, f"{clf_name}_SHAP_bar.png")
        plt.tight_layout()
        plt.savefig(bar_path, dpi=300, bbox_inches='tight')
        plt.close()
        print(f" -> SHAP bar plot salvato in {bar_path}")

        shap_array = shap_values_classe1.values
        # Top 3 feature in base all'importanza media (valori assoluti)
        top_features = np.abs(shap_array).mean(0).argsort()[-3:][::-1]
        # SHAP dependence plot
        for i in top_features:
            feature_name = X_test.columns[i]
            shap.dependence_plot(feature_name, shap_array, X_test, show=False)
            dep_path = os.path.join(output_dir, f"{clf_name}_SHAP_dependence_{feature_name}.png")
            plt.tight_layout()
            plt.savefig(dep_path, dpi=300)
            plt.close()
            print(f" -> SHAP dependence plot per '{feature_name}' salvato in {dep_path}")
    except Exception as e:
        print(f"[ERRORE SHAP] Impossibile generare il grafico per {clf_name}: {e}")


    return risultati_test

def evaluate_model(model, X, y, phase, output_dir, classifier_name):
    
    y_pred = model.predict(X)

    # Ottieni le probabilità se disponibili
    if hasattr(model, "predict_proba"):
        y_score = model.predict_proba(X)[:, 1]
    elif hasattr(model, "decision_function"):
        y_score = model.decision_function(X)
    else:
        y_score = None

    report = classification_report(y, y_pred, output_dict=True)
    matrix = confusion_matrix(y, y_pred)
    accuracy = report['accuracy']
    f1_score_classe1 = report['1.0']['f1-score']
    f1_score_classe0 = report['0.0']['f1-score']
    recall_0 = report['0.0']['recall']
    recall_1 = report['1.0']['recall']
    precision_0 = report['0.0']['precision']
    precision_1 = report['1.0']['precision']
    bal_accuracy = balanced_accuracy_score(y, y_pred)
    
    # Calcolo sicuro dell'AUC
    try:
        auc = roc_auc_score(y, y_score) if y_score is not None and len(np.unique(y)) > 1 else 'N/A'
    except Exception as e:
        print(f"Errore nel calcolo AUC per {classifier_name} in fase {phase}: {e}")
        auc = 'N/A'

    """if isinstance(auc, float):
        score = 0.5 * f1_score_classe1 + 0.3 * auc + 0.2 * bal_accuracy
    else:
        score = 0.7 * f1_score_classe1 + 0.3 * bal_accuracy  # ignora AUC"""

    
    # Salvataggio report in Excel
    pd.DataFrame(report).transpose().to_excel(
        os.path.join(output_dir, f"{classifier_name}_{phase}_report.xlsx"))
    cm_df = pd.DataFrame(matrix, columns=['Pred 0', 'Pred 1'], index=['True 0', 'True 1'])
    cm_df.transpose().to_excel(os.path.join(output_dir, f"{classifier_name}_{phase}_confusion_matrix.xlsx"))

    # Salvataggio plot
    plot_confusion_matrix(matrix, phase, classifier_name, output_dir)
    if y_score is not None and isinstance(auc, float):
        plot_roc_curve(y, y_score, phase, classifier_name, output_dir)
        plot_precision_recall(y, y_score, phase, classifier_name, output_dir)

    return {
        'classification_report': report,
        'confusion_matrix': matrix.tolist(),
        'roc_auc': auc,
        'accuracy': accuracy,
        'bal_accuracy': bal_accuracy,
        'F1_score_1': f1_score_classe1,
        'F1_score_0': f1_score_classe0,
        'precision_0': precision_0,
        'precision_1': precision_1,
        'recall_0': recall_0,
        'recall_1': recall_1,
        #'score': score
    }
    
def train_tuning_validation(X_train, y_train, X_val, y_val, feature_selection_name, output_dir):
    risultati = {}
    report_aggregato = []

    classifiers = {
        "Naive Bayes": GaussianNB(),
        "KNN": KNeighborsClassifier(),
        "SVM Linear": SVC(kernel='linear', probability=True),
        "SVM Poly": SVC(kernel='poly', probability=True),
        "SVM RBF": SVC(kernel='rbf', probability=True),
        "SVM Sigmoid": SVC(kernel='sigmoid', probability=True),
        "Decision Tree": DecisionTreeClassifier(random_state=42),
        "Bagging DT": BaggingClassifier(estimator=DecisionTreeClassifier(), random_state=42),
        "AdaBoost": AdaBoostClassifier(random_state=42),
        "Gradient Boosting": GradientBoostingClassifier(random_state=42),
        "Random Forest": RandomForestClassifier(random_state=42),
        "SGD": SGDClassifier(random_state=42)
    }

    param_grid = {
        "Naive Bayes": {},

        "KNN": {
            "n_neighbors": [3, 5, 7, 9],
            "weights": ['uniform', 'distance'],
            "p": [1, 2]  # 1 = Manhattan, 2 = Euclidea
        },

        "SVM Linear": {
        "C": [0.01, 0.1, 1, 10, 100],
        "gamma": ['scale', 'auto']
        },
        "SVM Poly": {
            "C": [0.1, 1, 10],
            "degree": [2, 3, 4],
            "gamma": ['scale', 'auto']
        },
        "SVM RBF": {
            "C": [0.1, 1, 10, 100],
            "gamma": [0.001, 0.01, 0.1, 'scale', 'auto']
        },
        "SVM Sigmoid": {
            "C": [0.1, 1, 10],
            "gamma": ['scale', 'auto']
        },

        "Decision Tree": {
            "max_depth": [None, 4, 6, 8, 10],
            "min_samples_split": [2, 5, 10],
            "criterion": ["gini", "entropy"]
        },

        "Bagging DT": {
            "n_estimators": [10, 50, 100],
            "max_samples": [0.5, 0.7, 1.0],
            "bootstrap": [True, False]
        },

        "AdaBoost": {
            "n_estimators": [50, 100, 200],
            "learning_rate": [0.01, 0.1, 0.5, 1.0]
        },

        "Gradient Boosting": {
            "n_estimators": [50, 100, 150],
            "learning_rate": [0.01, 0.05, 0.1],
            "max_depth": [3, 5, 7]
        },

        "Random Forest": {
            "n_estimators": [50, 100, 200],
            "max_depth": [None, 10, 20],
            "max_features": ['sqrt', 'log2'],
            "min_samples_split": [2, 5]
        },

        "SGD": {
            "alpha": [1e-5, 1e-4, 1e-3],
            "loss": ['hinge', 'log_loss'],
            "penalty": ['l2', 'l1', 'elasticnet']
        }
    }


    print(f"\nTUNING IPERPARAMETRI per metodo di feature selection: {feature_selection_name}")

    best_clf_for_fs={}
    best_auc = -np.inf
    best_clf_model = None
    best_clf_name = ""   
    
    for name, clf in classifiers.items():
        print(f"\nTraining {name} con tuning iperparametri...")

        classifier_dir = os.path.join(output_dir, name)
        os.makedirs(classifier_dir, exist_ok=True)

        cv_stats = StratifiedKFold(n_splits=5, shuffle=True, random_state=42)
        
        grid = GridSearchCV(clf, param_grid.get(name, {}), cv= cv_stats, n_jobs=-1, scoring='roc_auc', verbose=1)
        start_time = time.time()
        grid.fit(X_train, y_train)
        end_time = time.time()

        # calcolo e salvataggio parametri training
        best_model = grid.best_estimator_
        y_train_pred = best_model.predict(X_train)
        
        # Ottieni le probabilità se disponibili
        if hasattr(best_model, "predict_proba"):
            y_score = best_model.predict_proba(X_train)[:, 1]
        elif hasattr(best_model, "decision_function"):
            y_score = best_model.decision_function(X_train)
        else:
            y_score = None
        
        try:
            auc = roc_auc_score(y_train, y_score) if y_score is not None and len(np.unique(y_train)) > 1 else 'N/A'
        except Exception as e:
            print(f"Errore nel calcolo AUC per {name} in fase Training: {e}")
            auc = 'N/A'
            
        bal_accuracy_t = balanced_accuracy_score(y_train, y_train_pred)
        report_train = classification_report(y_train, y_train_pred, output_dict=True)
        matrix = confusion_matrix(y_train, y_train_pred)
        accuracy_t = report_train['accuracy']
        f1_score_classe1_t = report_train['1.0']['f1-score']
        f1_score_classe0_t = report_train['0.0']['f1-score']
        recall_0_t = report_train['0.0']['recall']
        recall_1_t = report_train['1.0']['recall']
        precision_0_t = report_train['0.0']['precision']
        precision_1_t = report_train['1.0']['precision']

        report_train_df = pd.DataFrame(report_train).transpose()
        report_train_df.to_excel(os.path.join(classifier_dir, f"{name}_training_report.xlsx"))
        train_results = {
            'classification_report': report_train,
            'confusion matrix': matrix.tolist(),
            'roc_auc': auc,
            'accuracy': accuracy_t,
            'bal_accuracy': bal_accuracy_t,
            'F1_score_1': f1_score_classe1_t,
            'F1_score_0': f1_score_classe0_t,
            'precision_0': precision_0_t,
            'precision_1': precision_1_t,
            'recall_0': recall_0_t,
            'recall_1': recall_1_t,
        }

        model = grid.best_estimator_
        params = grid.best_params_
        print(f"Tempo di training: {end_time - start_time:.2f} s")
        print(f"Migliori parametri per {name}: {params}")
        
        # Salvataggio modello
        joblib.dump(model, os.path.join(classifier_dir, f"{name}_model.pkl"))

        # Salvataggio parametri in un file txt
        param_txt_path = os.path.join(classifier_dir, f"{name}_best_params.txt")
        with open(param_txt_path, "w") as f:
            f.write(f"Best parameters for {name}:\n")
            for param, val in params.items():
                f.write(f"{param}: {val}\n")

        # Valutazione
        val_results = evaluate_model(model, X_val, y_val, 'validation', classifier_dir, name)
        #test_results = evaluate_model(best_model, X_test, y_test, 'test', classifier_dir, name)

        # QUI VA INSERITO IL SALVTAGGIO DEL CALSSIFICATORE MIGLIORE DA USARE POI PER IL TEST SET.
        if val_results['roc_auc'] > best_auc:
            best_auc = val_results['roc_auc'] 
            best_clf_model = model
            best_clf_name = name
            best_clf_params = params

            
        risultati[name] = {
            'train': train_results,
            'validation': val_results,
        }

        report_aggregato.append({
                'Classifier': name,
                'Train ROC AUC': train_results['roc_auc'],
                'Train Accuracy': train_results['accuracy'],
                'Train Bal. Accuracy': train_results['bal_accuracy'],
                'Train Precision_0': train_results['precision_0'],
                'Train Precision_1': train_results['precision_1'],
                'Train Recall_0': train_results['recall_0'],
                'Train Recall_1': train_results['recall_1'],
                'Train F1 score 0': train_results['F1_score_0'],
                'Train F1 score 1': train_results['F1_score_1'],
                'Val ROC AUC': val_results['roc_auc'],
                'Val Accuracy': val_results['accuracy'],
                'Val Bal. Accuracy': val_results['bal_accuracy'],
                'Val Precision_0': val_results['precision_0'],
                'Val Precision_1': val_results['precision_1'],
                'Val Recall_0': val_results['recall_0'],
                'Val Recall_1': val_results['recall_1'],
                'Val F1 score 0': val_results['F1_score_0'],
                'Val F1 score 1': val_results['F1_score_1'],
            })
        
    best_clf_for_fs = {
    "model": best_clf_model,
    "best auc": best_auc,
    "classif_name": best_clf_name,
    "parametri": best_clf_params
    }
    
    best_clf_for_fs_to_save = {
        'model': str(best_clf_model),
        'best auc': best_auc,
        'classif_name':best_clf_name,
        'parametri': best_clf_params
    }
    with open(os.path.join(output_dir, f"best_classifier_{feature_selection_name}.txt"), "w") as f:
        json.dump(best_clf_for_fs_to_save, f, indent=4)
    
    # Salvataggio del report aggregato
    df_report = pd.DataFrame(report_aggregato)
    report_path = os.path.join(output_dir, f"report_aggregato_{feature_selection_name}.xlsx")
    df_report.to_excel(report_path, index=False)

    return risultati, best_clf_for_fs


def esegui_classificatori_per_feature_selection_da_cartella(
    train_path, val_path, test_path, cartella_feature_selection, output_dir_base):
    
    # Caricamento dataset
    train_df = pd.read_csv(train_path, index_col=0)
    val_df = pd.read_csv(val_path, index_col=0)
    test_df = pd.read_csv(test_path, index_col=0)
    
    y_train = train_df['Label']
    y_val = val_df['Label']
    y_test = test_df['Label']

    # Prendi tutti i file xlsx di feature selection
    feature_files = [
        f for f in os.listdir(cartella_feature_selection)
        if f.endswith('.xlsx')
    ]

    summary_globale = []
    lista_risultati = []
    clf_for_cvext = {}
    dizionario_feature_selzionate = {}
    
    for feat_file in feature_files:
        metodo_nome = os.path.splitext(feat_file)[0]
        print(f"\nMetodo feature selection: {metodo_nome}")
        
        output_dir = os.path.join(output_dir_base, metodo_nome)
        os.makedirs(output_dir, exist_ok=True)

        feat_path = os.path.join(cartella_feature_selection, feat_file)
        df_feat = pd.read_excel(feat_path)

        # Estrai i nomi delle feature
        selected_features = df_feat.iloc[:, 0].dropna().tolist()
        dizionario_feature_selzionate[metodo_nome] = selected_features
        
        # Controllo che le feature siano tutte nel dataset
        missing_features = [f for f in selected_features if f not in train_df.columns]
        if missing_features:
            print(f"Feature mancanti nel dataset e saltate: {missing_features}")
            continue  # Salta al metodo successivo

        # Estrazione subset dei dati
        X_train = train_df[selected_features]
        X_val = val_df[selected_features]
        X_test = test_df[selected_features]             
                
        # Training e valutazione( ho tolto come parametri  X_test, y_test)
        risultati, best_clf_for_fs = train_tuning_validation(
            X_train, y_train, X_val, y_val, metodo_nome, output_dir
        )
        
        #clf_for_cvext[metodo_nome] = best_clf_for_fs
        clf_model = best_clf_for_fs['model']
        clf_name = best_clf_for_fs['classif_name']
        clf_for_cvext[metodo_nome] = {clf_name: clf_model}

        risultati_test = test_and_SHAP(
            model=best_clf_for_fs["model"],
            clf_name=best_clf_for_fs["classif_name"],
            feature_selection_name = metodo_nome,
            sel_features = selected_features,
            X_test=X_test,
            y_test=y_test,
            output_dir=output_dir
        )

        plot_and_save_results(risultati, output_dir, metodo_nome)
        
        risultati['test']=risultati_test
        lista_risultati.append(risultati_test)
        
        # Aggiunta al report aggregato
        for classifier_name, res in risultati.items():
            if classifier_name == 'test':
                summary_globale.append({
                    'Test ROC AUC': res['roc_auc'],
                    'Test Accuracy': res['accuracy'],
                    'Test Bal. Accuracy': res['bal_accuracy'],
                    'Test Precision_0': res['precision_0'],
                    'Test Precision_1': res['precision_1'],
                    'Test Recall_0': res['recall_0'],
                    'Test Recall_1': res['recall_1'],
                    'Test F1 score 0': res['F1_score_0'],
                    'Test F1 score 1': res['F1_score_1']
                    })
            else:
                summary_globale.append({
                'Metodo Feature Selection': metodo_nome,
                'Classifier': classifier_name,
                'Train ROC AUC': res['train']['roc_auc'],
                'Train Accuracy': res['train']['accuracy'],
                'Train Bal. Accuracy': res['train']['bal_accuracy'],
                'Train Precision_0': res['train']['precision_0'],
                'Train Precision_1': res['train']['precision_1'],
                'Train Recall_0': res['train']['recall_0'],
                'Train Recall_1': res['train']['recall_1'],
                'Train F1 score 0': res['train']['F1_score_0'],
                'Train F1 score 1': res['train']['F1_score_1'],
                'Val ROC AUC': res['validation']['roc_auc'],
                'Val Accuracy': res['validation']['accuracy'],
                'Val Bal. Accuracy': res['validation']['bal_accuracy'],
                'Val Precision_0': res['validation']['precision_0'],
                'Val Precision_1': res['validation']['precision_1'],
                'Val Recall_0': res['validation']['recall_0'],
                'Val Recall_1': res['validation']['recall_1'],
                'Val F1 score 0': res['validation']['F1_score_0'],
                'Val F1 score 1': res['validation']['F1_score_1'],
            })
    
    # cross validation esterna per valutare le metriche con le relative std
    
    try:
        external_crossval_with_fs(train_path, val_path, test_path, clf_for_cvext,
                               dizionario_feature_selzionate, output_dir_base, n_splits=5, random_state=42)
    except Exception as e:
            logging.exception(f"Errore nel processo di cross validation esterna")
             
    # Salvataggio dei file
    df_finale = pd.DataFrame(lista_risultati)
    path_finale = os.path.join(output_dir_base, "tutti_i_risultati_dei_test.xlsx")
    df_finale.to_excel(path_finale, index=False)
    
    plot_risultati_test(df_finale, output_dir_base)
    
    summary_path = os.path.join(output_dir_base, "summary_auc_completo.xlsx")
    pd.DataFrame(summary_globale).to_excel(summary_path, index=False)
    print(f"\n Report globale salvato in: {summary_path}")
